% This function gets the name of a file containing a spectrum in
% comma-separated format (wavelength,intensity), and a string representing
% the desired form of a fit to the spectrum.

% The fit string can be for a specific mineral, if the form of the spectrum
% is known (e.g., 'GOE' for goethite, which can be fit by three Voigt
% peaks).

% For a new fit type, the string lists the number of S-functions,
% Z-functions, T-functions (tables) and Gaussian peaks. For example, a fit
% represented by the string 'S2Z0T0G3' is composed of two S-functions and 3
% Gaussian peaks. The user is prompted for limits to the parameter values.

% Input:
%   fileName - a string with the path to a csv file with the spectrum
%              (wavelength,intensity).
%   fitStr - a string representing the fit type ('GOE', 'HEM', 'MAG' or a
%            user-defined number of S, Z and T functions and Voigt peaks.
%   fplot - a flag for plotting the deconvolved spectrum (0 or 1).
%
% Output:
%   fitRes - best-fit parameters
%            For S-functions
%            S(1) - S-function intensity
%            S(2) - S-function rise onset wavelength
%            S(3) - S-function rise end wavelength
%            For Z-functions
%            Z(1) - Z-function intensity
%            Z(2) - Z-function fall onset wavelength
%            Z(3) - Z-function fall end wavelength
%            For T-functions
%            T(1) - T-function intensity
%            T(2) - T-function plateau center wavelength
%            T(3) - T-function plateau width
%            T(4) - T-function dropoff steepness
%            For Gaussians
%            G(1) - Gaussian peak intensity
%            G(2) - Gaussian full width and half-maximum
%            G(3) - Gaussian peak center wavelength
%   RGBres - RGB color triplet representation of the true color of the
%            deconvolved spectrum.
%   err - Sum of squared errors between measured and modeled spectrum.

function [fitRes,RGBres,err] = fitSpecFinal(fileName,fitStr,fplot)

S1 = []; S2 = []; S3 = [];
Z1 = []; Z2 = []; Z3 = [];
T1 = []; T2 = []; T3 = []; T4 = [];
G1 = []; G2 = [];

switch fitStr
    case 'GOE'
        % Three Gaussian peaks. Each has a peak intensity (G1), peak width
        % (G2), peak position (G3).
        G1 = [0.0,1.0; 0.0,0.7; 0.5,1.0];
        G2 = [0,200; 0,200; 0,200];
        G3 = [420,520; 550,630; 650,750];
    case 'HEM'
        % Three Gaussian peaks. Each has a peak intensity (G1), peak width
        % (G2), peak position (G3). These are guided by our results.
        G1 = [0.0,1.0; 0.0,1.0; 0.0,1.0]; % GOOD
        G2 = [0,200; 0,200; 0,200];
        G3 = [420,500; 550,640; 650,740];
    case 'MAG' % UPDATE THIS
        % Three Gaussian peaks. Each has a peak intensity (G1), peak width
        % (G2), peak position (G3).
        G1 = [0.0,1.0; 0.0,1.0; 0.0,1.0];
        G2 = [0,100; 0,100; 0,200];
        G3 = [420,470; 480,600; 610,750];
    otherwise
        % S-function
        nS = str2num(fitStr(2));
        for i = 1:nS
            I = input(sprintf('S-function %i intensity limits? [low,high] ',i));
            S1 = cat(1,S1,I);
            Si = input(sprintf('S-function %i rise onset position limits? [low,high] ',i));
            S2 = cat(1,S2,Si);
            Sf = input(sprintf('S-function %i rise end position limits? [low,high] ',i));
            S3 = cat(1,S3,Sf);
        end
            
        % Z-function
        nZ = str2num(fitStr(4));
        for i = 1:nZ
            I = input(sprintf('Z-function %i intensity limits? [low,high] ',i));
            Z1 = cat(1,Z1,I);
            Zi = input(sprintf('Z-function %i fall onset position limits? [low,high] ',i));
            Z2 = cat(1,Z2,Zi);
            Zf = input(sprintf('Z-function %i fall end position limits? [low,high] ',i));
            Z3 = cat(1,Z3,Zf);
        end
        
        % T-function
        nT = str2num(fitStr(6));
        for i = 1:nT
            I = input(sprintf('T-function %i intensity limits? [low,high] ',i));
            T1 = cat(1,T1,I);
            Ts = input(sprintf('T-function %i center position limits? [low,high] ',i));
            T2 = cat(1,T2,Ts);
            Ts = input(sprintf('T-function %i plateau width limits? [low,high] ',i));
            T3 = cat(1,T3,Ts);
            Ts = input(sprintf('T-function %i dropoff steepness limits? [low,high] ',i));
            T4 = cat(1,T4,Ts); % Higher T4 means faster dropoff
        end
        
        % Gaussian peak
        nG = str2num(fitStr(8));
        for i = 1:nG
            I = input(sprintf('Gaussian %i peak intensity limits? ',i));
            G1 = cat(1,G1,I);
            W = input(sprintf('Gaussian %i peak width limits? ',i));
            G2 = cat(1,G2,W);
            P = input(sprintf('Gaussian %i peak position limits? ',i));
            G3 = cat(1,G3,P);
        end
end

nS = length(S1); nZ = length(Z1); nT = length(T1); nG = length(G1);

data = load(fileName);
L = data(:,1);
I = data(:,2); I(I<0) = 0;
Imm = movmean(I,21); Ims = movstd(I,21);
L = L(abs(I-Imm)<(2.*Ims));
I = I(abs(I-Imm)<(2.*Ims));
I = I./max(I);
I = I(L<=800); L = L(L<=800);

pLo = []; pHi = []; pSt = [];
if ~isempty(S1)
    pLo = cat(1,pLo,[S1(:,1); S2(:,1); S3(:,1)]);
    pHi = cat(1,pHi,[S1(:,2); S2(:,2); S3(:,2)]);
else
    pLo = cat(1,pLo,[[]; []; [];]);
    pHi = cat(1,pHi,[[]; []; [];]);
end

if ~isempty(Z1)
    pLo = cat(1,pLo,[Z1(:,1); Z2(:,1); Z3(:,1)]);
    pHi = cat(1,pHi,[Z1(:,2); Z2(:,2); Z3(:,2)]);
else
    pLo = cat(1,pLo,[[]; []; [];]);
    pHi = cat(1,pHi,[[]; []; [];]);
end

if ~isempty(T1)
    pLo = cat(1,pLo,[T1(:,1); T2(:,1); T3(:,1); T4(:,1)]);
    pHi = cat(1,pHi,[T1(:,2); T2(:,2); T3(:,2); T4(:,2)]);
else
    pLo = cat(1,pLo,[[]; []; []; [];]);
    pHi = cat(1,pHi,[[]; []; []; [];]);
end

if ~isempty(G1)
    pLo = cat(1,pLo,[G1(:,1); G2(:,1); G3(:,1)]);
    pHi = cat(1,pHi,[G1(:,2); G2(:,2); G3(:,2)]);
else
    pLo = cat(1,pLo,[[]; []; []; [];]);
    pHi = cat(1,pHi,[[]; []; []; [];]);
end

options = optimset('Display','none','MaxFunEvals',1e6,'TolFun',1e-64,'TolX',1e-64);

pSt = (pLo + pHi)./2;
[fitRes,err] = fmincon(@(p) minFun(p,nS,nZ,nT,nG,L,I),pSt,[],[],[],[],pLo,pHi,[],options);

nStarts = 2e1;
for i = 1:nStarts
    pSt = random('Uniform',pLo,pHi);
    [fitResTemp,errTemp] = fmincon(@(p) minFun(p,nS,nZ,nT,nG,L,I),pSt,[],[],[],[],pLo,pHi,[],options);
    if errTemp <= err
        err = errTemp;
        fitRes = fitResTemp;
    end
end

Lcalc = ((min(L)/1.1):0.5:(max(L)*1.1))'; % nx1
Icalc = fitFun(fitRes,nS,nZ,nT,nG,Lcalc);

[lMatch, xFcn, yFcn, zFcn] = colorMatchFcn('cie_1931');
xyz = interp1(lMatch', [xFcn; yFcn; zFcn]', L, 'pchip', 0);
XYZ = sum(xyz.*repmat(I,1,3),1)./sum(I);

C = makecform('xyz2srgb');
RGBres = applycform(XYZ,C);

if fplot == 1
    hold off
    if fitRes(4) < 35 % W1 (I am guessing that these are spectra with shoulders)
        C = [0.7,0.7,1.0];
    else
        C = [1.0,0.9,0.7];
    end
    C = RGBres.*1.8; C(C>1) = 1;
    patch([Lcalc; flipud(Lcalc)],[Icalc; zeros(size(Icalc))],C,'EdgeColor','none')
    hold on
    plot(L,I,'k.','MarkerSize',8)
    for iG = 1:nG
        %        I                           w             l
        fG = fitRes(iG).*gaussmf(Lcalc,[fitRes(iG+nG),fitRes(iG+2*nG)]);
        plot(Lcalc,fG,'Color',[0.6,0.6,0.6],'LineWidth',1)
    end
    set(gca,'FontSize',14,'YTick',[]);
    xlabel('Wavelength (nm)','FontSize',16)
    ylabel('Relative intensity','FontSize',16)
    xlim([ceil(min(Lcalc)),floor(max(Lcalc))])
    xl = xlim; yl = ylim;
    text(xl(2),yl(2),sprintf('%1.1e',err),'FontSize',16,'HorizontalAlignment','right','VerticalAlignment','top')
    
    drawnow
end

fitRes = fitRes'; % [1x9]

end

function f = minFun(p,nS,nZ,nT,nG,L,Im)

Ic = fitFun(p,nS,nZ,nT,nG,L);
% f = sum(abs(Ic - Im));
f = sum((Ic - Im).^2);
end

function Ic = fitFun(p,nS,nZ,nT,nG,L)
if nS > 0
    S1 = p((0*nS+1):(1*nS));
    S2 = p((1*nS+1):(2*nS));
    S3 = p((2*nS+1):(3*nS));
    p = p((3*nS+1):end);
end

if nZ > 0
    Z1 = p((0*nZ+1):(1*nZ));
    Z2 = p((1*nZ+1):(2*nZ));
    Z3 = p((2*nZ+1):(3*nZ));
    p = p((3*nZ+1):end);
end

if nT > 0
    T1 = p((0*nT+1):(1*nT));
    T2 = p((1*nT+1):(2*nT));
    T3 = p((2*nT+1):(3*nT));
    T4 = p((3*nT+1):(4*nT));
    p = p((4*nT+1):end);
end

if nG > 0
    G1 = p((0*nG+1):(1*nG));
    G2 = p((1*nG+1):(2*nG));
    G3 = p((2*nG+1):(3*nG));
    p = p((3*nG+1):end);
end

fS = 0;
for i = 1:nS
    fS = fS + S1(i).*smf(L,[S2(i),S3(i)]);
end
fZ = 0;
for i = 1:nZ
    fZ = fZ + Z1(i).*smf(L,[Z2(i),Z3(i)]);
end
fT = 0;
for i = 1:nT
    fT = fT + T1(i).*gbellmf(L,[T3(i),T4(i),T2(i)]);
end
fG = 0;
for i = 1:nG
    fG = fG + G1(i).*gaussmf(L,[G2(i),G3(i)]);
end
Ic = fS + fZ + fT + fG;
end
